<?php
/**
 * Join Game Room
 * POST /api/rooms/join
 */

require_once '../../config/config.php';
require_once '../../config/database.php';
require_once '../../config/jwt.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    error('Method not allowed', 'METHOD_NOT_ALLOWED', 405);
}

try {
    $user_id = getCurrentUser();
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        error('Invalid JSON input', 'INVALID_INPUT');
    }
    
    // Validate required fields
    validateRequired($input, ['room_code']);
    
    $room_code = sanitizeInput($input['room_code']);
    
    $db = new Database();
    $conn = $db->getConnection();
    
    // Find room
    $stmt = $conn->prepare("
        SELECT r.*, u.username as creator_username 
        FROM rooms r
        JOIN users u ON r.creator_id = u.id
        WHERE r.room_code = ? AND r.status = 'waiting'
    ");
    $stmt->execute([$room_code]);
    $room = $stmt->fetch();
    
    if (!$room) {
        error('Room not found or not available', 'ROOM_NOT_FOUND');
    }
    
    // Check if room is expired
    if (strtotime($room['expires_at']) < time()) {
        error('Room has expired', 'ROOM_EXPIRED');
    }
    
    // Check if user is trying to join their own room
    if ($room['creator_id'] == $user_id) {
        error('Cannot join your own room', 'INVALID_JOIN');
    }
    
    // Check user's balance
    $stmt = $conn->prepare("SELECT main_balance, bonus_balance FROM wallets WHERE user_id = ?");
    $stmt->execute([$user_id]);
    $wallet = $stmt->fetch();
    
    if (!$wallet) {
        error('Wallet not found', 'WALLET_NOT_FOUND', 404);
    }
    
    $total_balance = $wallet['main_balance'] + $wallet['bonus_balance'];
    if ($total_balance < $room['stake_amount']) {
        error('Insufficient balance', 'INSUFFICIENT_BALANCE');
    }
    
    // Start transaction
    $conn->beginTransaction();
    
    try {
        // Update room status
        $stmt = $conn->prepare("UPDATE rooms SET status = 'active' WHERE id = ?");
        $stmt->execute([$room['id']]);
        
        // Create game
        $stmt = $conn->prepare("
            INSERT INTO games (room_id, player1_id, player2_id, stake_amount, status) 
            VALUES (?, ?, ?, ?, 'active')
        ");
        $stmt->execute([$room['id'], $room['creator_id'], $user_id, $room['stake_amount']]);
        $game_id = $conn->lastInsertId();
        
        // Deduct stake from both players
        $stake_amount = $room['stake_amount'];
        
        // Deduct from creator
        $stmt = $conn->prepare("
            UPDATE wallets 
            SET main_balance = main_balance - ? 
            WHERE user_id = ?
        ");
        $stmt->execute([$stake_amount, $room['creator_id']]);
        
        // Deduct from joiner
        $stmt = $conn->prepare("
            UPDATE wallets 
            SET main_balance = main_balance - ? 
            WHERE user_id = ?
        ");
        $stmt->execute([$stake_amount, $user_id]);
        
        // Create transactions
        $stmt = $conn->prepare("
            INSERT INTO transactions (user_id, type, amount, balance_type, description, status) 
            VALUES (?, 'game_loss', ?, 'main', 'Game stake - Room: " . $room_code . "', 'completed')
        ");
        $stmt->execute([$room['creator_id'], $stake_amount]);
        $stmt->execute([$user_id, $stake_amount]);
        
        $conn->commit();
        
        success([
            'game' => [
                'id' => $game_id,
                'room_code' => $room_code,
                'stake_amount' => $stake_amount,
                'player1' => [
                    'id' => $room['creator_id'],
                    'username' => $room['creator_username']
                ],
                'player2' => [
                    'id' => $user_id,
                    'username' => 'You'
                ],
                'status' => 'active'
            ]
        ], 'Successfully joined room');
        
    } catch (Exception $e) {
        $conn->rollBack();
        throw $e;
    }
    
} catch (Exception $e) {
    error_log("Join room error: " . $e->getMessage());
    error('Failed to join room', 'JOIN_ERROR', 500);
}
?>
