<?php
/**
 * Create Flutterwave Payment Charge
 * POST /api/pay/create_charge
 */

require_once '../../config/config.php';
require_once '../../config/database.php';
require_once '../../config/jwt.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    error('Method not allowed', 'METHOD_NOT_ALLOWED', 405);
}

try {
    $user_id = getCurrentUser();
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        error('Invalid JSON input', 'INVALID_INPUT');
    }
    
    // Validate required fields
    validateRequired($input, ['amount']);
    
    $amount = floatval($input['amount']);
    
    // Validate amount
    if ($amount < MIN_DEPOSIT) {
        error('Minimum deposit amount is ' . MIN_DEPOSIT, 'INVALID_AMOUNT');
    }
    
    if ($amount > 1000000) { // Max 1M TZS
        error('Maximum deposit amount is 1,000,000 TZS', 'INVALID_AMOUNT');
    }
    
    $db = new Database();
    $conn = $db->getConnection();
    
    // Get user details
    $stmt = $conn->prepare("SELECT username, email, phone FROM users WHERE id = ?");
    $stmt->execute([$user_id]);
    $user = $stmt->fetch();
    
    if (!$user) {
        error('User not found', 'USER_NOT_FOUND', 404);
    }
    
    // Generate transaction reference
    $tx_ref = 'CHK_' . time() . '_' . $user_id;
    
    // Create pending transaction
    $conn->beginTransaction();
    
    try {
        $stmt = $conn->prepare("
            INSERT INTO transactions (user_id, type, amount, balance_type, description, reference, status) 
            VALUES (?, 'deposit', ?, 'main', 'Flutterwave deposit', ?, 'pending')
        ");
        $stmt->execute([$user_id, $amount, $tx_ref]);
        $transaction_id = $conn->lastInsertId();
        
        $conn->commit();
        
        // Create Flutterwave payment data
        $payment_data = [
            'tx_ref' => $tx_ref,
            'amount' => $amount,
            'currency' => 'TZS',
            'redirect_url' => SITE_URL . '/api/pay/callback.php',
            'customer' => [
                'email' => $user['email'],
                'name' => $user['username'],
                'phone_number' => $user['phone']
            ],
            'customizations' => [
                'title' => 'Checkers Game Deposit',
                'description' => 'Deposit to your Checkers Game wallet',
                'logo' => SITE_URL . '/assets/logo.png'
            ],
            'meta' => [
                'user_id' => $user_id,
                'transaction_id' => $transaction_id
            ]
        ];
        
        // For shared hosting, we'll return the payment data
        // The mobile app will handle the Flutterwave integration
        success([
            'transaction_id' => $transaction_id,
            'tx_ref' => $tx_ref,
            'amount' => $amount,
            'currency' => 'TZS',
            'payment_data' => $payment_data,
            'flutterwave_public_key' => FLUTTERWAVE_PUBLIC_KEY,
            'redirect_url' => SITE_URL . '/api/pay/callback.php'
        ], 'Payment initialized');
        
    } catch (Exception $e) {
        $conn->rollBack();
        throw $e;
    }
    
} catch (Exception $e) {
    error_log("Create charge error: " . $e->getMessage());
    error('Failed to create payment', 'PAYMENT_ERROR', 500);
}
?>
