<?php
/**
 * Quick Match (Pay Now)
 * POST /api/match/quick
 */

require_once '../../config/config.php';
require_once '../../config/database.php';
require_once '../../config/jwt.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    error('Method not allowed', 'METHOD_NOT_ALLOWED', 405);
}

try {
    $user_id = getCurrentUser();
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        error('Invalid JSON input', 'INVALID_INPUT');
    }
    
    // Validate required fields
    validateRequired($input, ['stake_amount']);
    
    $stake_amount = floatval($input['stake_amount']);
    
    // Validate stake amount
    if ($stake_amount < 100) {
        error('Minimum stake amount is 100 TZS', 'INVALID_STAKE');
    }
    
    if ($stake_amount > 100000) {
        error('Maximum stake amount is 100,000 TZS', 'INVALID_STAKE');
    }
    
    $db = new Database();
    $conn = $db->getConnection();
    
    // Check user's balance
    $stmt = $conn->prepare("SELECT main_balance, bonus_balance FROM wallets WHERE user_id = ?");
    $stmt->execute([$user_id]);
    $wallet = $stmt->fetch();
    
    if (!$wallet) {
        error('Wallet not found', 'WALLET_NOT_FOUND', 404);
    }
    
    $total_balance = $wallet['main_balance'] + $wallet['bonus_balance'];
    if ($total_balance < $stake_amount) {
        error('Insufficient balance', 'INSUFFICIENT_BALANCE');
    }
    
    // Look for existing quick match queue
    $stmt = $conn->prepare("
        SELECT g.*, u.username as player1_username 
        FROM games g
        JOIN users u ON g.player1_id = u.id
        WHERE g.is_bot_game = 0 AND g.player2_id IS NULL 
        AND g.stake_amount = ? AND g.status = 'waiting'
        AND g.created_at > DATE_SUB(NOW(), INTERVAL 10 SECOND)
        ORDER BY g.created_at ASC
        LIMIT 1
    ");
    $stmt->execute([$stake_amount]);
    $existing_game = $stmt->fetch();
    
    $conn->beginTransaction();
    
    try {
        if ($existing_game) {
            // Join existing game
            $stmt = $conn->prepare("
                UPDATE games 
                SET player2_id = ?, status = 'active' 
                WHERE id = ?
            ");
            $stmt->execute([$user_id, $existing_game['id']]);
            
            // Deduct stake from both players
            $stmt = $conn->prepare("
                UPDATE wallets 
                SET main_balance = main_balance - ? 
                WHERE user_id = ?
            ");
            $stmt->execute([$stake_amount, $existing_game['player1_id']]);
            $stmt->execute([$stake_amount, $user_id]);
            
            // Create transactions
            $stmt = $conn->prepare("
                INSERT INTO transactions (user_id, type, amount, balance_type, description, status) 
                VALUES (?, 'game_loss', ?, 'main', 'Quick match stake', 'completed')
            ");
            $stmt->execute([$existing_game['player1_id'], $stake_amount]);
            $stmt->execute([$user_id, $stake_amount]);
            
            $game_id = $existing_game['id'];
            $is_bot_game = false;
            
        } else {
            // Create new game and wait for opponent or start bot game
            $stmt = $conn->prepare("
                INSERT INTO games (player1_id, stake_amount, status, is_bot_game) 
                VALUES (?, ?, 'waiting', 0)
            ");
            $stmt->execute([$user_id, $stake_amount]);
            $game_id = $conn->lastInsertId();
            
            // Deduct stake from player
            $stmt = $conn->prepare("
                UPDATE wallets 
                SET main_balance = main_balance - ? 
                WHERE user_id = ?
            ");
            $stmt->execute([$stake_amount, $user_id]);
            
            // Create transaction
            $stmt = $conn->prepare("
                INSERT INTO transactions (user_id, type, amount, balance_type, description, status) 
                VALUES (?, 'game_loss', ?, 'main', 'Quick match stake', 'completed')
            ");
            $stmt->execute([$user_id, $stake_amount]);
            
            // Start bot game after 10 seconds if no opponent joins
            $is_bot_game = true;
        }
        
        $conn->commit();
        
        success([
            'game' => [
                'id' => $game_id,
                'stake_amount' => $stake_amount,
                'is_bot_game' => $is_bot_game,
                'status' => $is_bot_game ? 'waiting' : 'active',
                'message' => $is_bot_game ? 
                    'Waiting for opponent... Bot will join in 10 seconds if no one joins' : 
                    'Match found! Game starting...'
            ]
        ], 'Quick match initiated');
        
    } catch (Exception $e) {
        $conn->rollBack();
        throw $e;
    }
    
} catch (Exception $e) {
    error_log("Quick match error: " . $e->getMessage());
    error('Failed to start quick match', 'QUICK_MATCH_ERROR', 500);
}
?>
