<?php
/**
 * Make Game Move
 * POST /api/game/:id/move
 */

require_once '../../config/config.php';
require_once '../../config/database.php';
require_once '../../config/jwt.php';
require_once '../game/checkers_engine.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    error('Method not allowed', 'METHOD_NOT_ALLOWED', 405);
}

try {
    $user_id = getCurrentUser();
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        error('Invalid JSON input', 'INVALID_INPUT');
    }
    
    // Validate required fields
    validateRequired($input, ['from', 'to']);
    
    $from = sanitizeInput($input['from']);
    $to = sanitizeInput($input['to']);
    
    // Get game ID from URL
    $path = $_SERVER['REQUEST_URI'];
    preg_match('/\/api\/game\/(\d+)\/move/', $path, $matches);
    $game_id = intval($matches[1] ?? 0);
    
    if (!$game_id) {
        error('Invalid game ID', 'INVALID_GAME_ID');
    }
    
    $db = new Database();
    $conn = $db->getConnection();
    
    // Get game details
    $stmt = $conn->prepare("
        SELECT g.*, u1.username as player1_username, u2.username as player2_username
        FROM games g
        LEFT JOIN users u1 ON g.player1_id = u1.id
        LEFT JOIN users u2 ON g.player2_id = u2.id
        WHERE g.id = ?
    ");
    $stmt->execute([$game_id]);
    $game = $stmt->fetch();
    
    if (!$game) {
        error('Game not found', 'GAME_NOT_FOUND', 404);
    }
    
    // Check if user is part of this game
    if ($game['player1_id'] != $user_id && $game['player2_id'] != $user_id) {
        error('You are not part of this game', 'NOT_PART_OF_GAME', 403);
    }
    
    // Check game status
    if ($game['status'] !== 'active') {
        error('Game is not active', 'GAME_NOT_ACTIVE');
    }
    
    // Determine player number
    $player_number = ($game['player1_id'] == $user_id) ? 1 : 2;
    
    // Load game state
    $board_state = json_decode($game['board_state'] ?? '[]', true);
    $current_player = $game['current_player'] ?? 1;
    
    // Check if it's the player's turn
    if ($current_player != $player_number) {
        error('Not your turn', 'NOT_YOUR_TURN');
    }
    
    // Initialize checkers engine
    $engine = new CheckersEngine();
    if (!empty($board_state)) {
        $engine->loadGameState($board_state, $current_player);
    }
    
    // Make the move
    $move_result = $engine->makeMove($from, $to);
    
    if (!$move_result['valid']) {
        error($move_result['message'], 'INVALID_MOVE');
    }
    
    // Get updated game state
    $game_state = $engine->getGameState();
    
    // Start transaction
    $conn->beginTransaction();
    
    try {
        // Update game state
        $stmt = $conn->prepare("
            UPDATE games 
            SET board_state = ?, current_player = ?, status = ?, winner_id = ?
            WHERE id = ?
        ");
        $stmt->execute([
            json_encode($game_state['board']),
            $game_state['currentPlayer'],
            $game_state['gameOver'] ? 'completed' : 'active',
            $game_state['winner'],
            $game_id
        ]);
        
        // Record the move
        $stmt = $conn->prepare("
            SELECT COUNT(*) as move_count FROM moves WHERE game_id = ?
        ");
        $stmt->execute([$game_id]);
        $move_count = $stmt->fetch()['move_count'] + 1;
        
        $stmt = $conn->prepare("
            INSERT INTO moves (game_id, player_id, move_number, from_position, to_position) 
            VALUES (?, ?, ?, ?, ?)
        ");
        $stmt->execute([$game_id, $user_id, $move_count, $from, $to]);
        
        // Handle game completion
        if ($game_state['gameOver']) {
            $winner_id = $game_state['winner'];
            $stake_amount = $game['stake_amount'];
            $platform_fee = ($stake_amount * 2 * PLATFORM_FEE) / 100;
            $winner_amount = ($stake_amount * 2) - $platform_fee;
            
            // Award winnings to winner
            $stmt = $conn->prepare("
                UPDATE wallets 
                SET main_balance = main_balance + ? 
                WHERE user_id = ?
            ");
            $stmt->execute([$winner_amount, $winner_id]);
            
            // Create winner transaction
            $stmt = $conn->prepare("
                INSERT INTO transactions (user_id, type, amount, balance_type, description, status) 
                VALUES (?, 'game_win', ?, 'main', 'Game winnings', 'completed')
            ");
            $stmt->execute([$winner_id, $winner_amount]);
        }
        
        $conn->commit();
        
        success([
            'move' => [
                'from' => $from,
                'to' => $to,
                'valid' => true
            ],
            'game_state' => [
                'board' => $game_state['board'],
                'current_player' => $game_state['currentPlayer'],
                'game_over' => $game_state['gameOver'],
                'winner' => $game_state['winner']
            ]
        ], 'Move successful');
        
    } catch (Exception $e) {
        $conn->rollBack();
        throw $e;
    }
    
} catch (Exception $e) {
    error_log("Game move error: " . $e->getMessage());
    error('Failed to make move', 'MOVE_ERROR', 500);
}
?>
