<?php
/**
 * User Registration API
 * POST /api/auth/register
 */

require_once '../../config/config.php';
require_once '../../config/database.php';
require_once '../../config/jwt.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    error('Method not allowed', 'METHOD_NOT_ALLOWED', 405);
}

try {
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        error('Invalid JSON input', 'INVALID_INPUT');
    }
    
    // Validate required fields
    validateRequired($input, ['username', 'email', 'password', 'phone']);
    
    // Sanitize input
    $username = sanitizeInput($input['username']);
    $email = sanitizeInput($input['email']);
    $password = $input['password'];
    $phone = sanitizeInput($input['phone']);
    $referral_code = sanitizeInput($input['referral_code'] ?? '');
    
    // Validate email format
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        error('Invalid email format', 'INVALID_EMAIL');
    }
    
    // Validate password strength
    if (strlen($password) < 6) {
        error('Password must be at least 6 characters', 'WEAK_PASSWORD');
    }
    
    // Validate username
    if (strlen($username) < 3 || strlen($username) > 20) {
        error('Username must be 3-20 characters', 'INVALID_USERNAME');
    }
    
    if (!preg_match('/^[a-zA-Z0-9_]+$/', $username)) {
        error('Username can only contain letters, numbers, and underscores', 'INVALID_USERNAME');
    }
    
    $db = new Database();
    $conn = $db->getConnection();
    
    // Check if username or email already exists
    $stmt = $conn->prepare("SELECT id FROM users WHERE username = ? OR email = ?");
    $stmt->execute([$username, $email]);
    if ($stmt->fetch()) {
        error('Username or email already exists', 'USER_EXISTS');
    }
    
    // Check referral code if provided
    $referred_by = null;
    if (!empty($referral_code)) {
        $stmt = $conn->prepare("SELECT id FROM users WHERE referral_code = ?");
        $stmt->execute([$referral_code]);
        $referrer = $stmt->fetch();
        if (!$referrer) {
            error('Invalid referral code', 'INVALID_REFERRAL');
        }
        $referred_by = $referrer['id'];
    }
    
    // Generate unique referral code
    do {
        $new_referral_code = generateReferralCode();
        $stmt = $conn->prepare("SELECT id FROM users WHERE referral_code = ?");
        $stmt->execute([$new_referral_code]);
    } while ($stmt->fetch());
    
    // Hash password
    $password_hash = hashPassword($password);
    
    // Start transaction
    $conn->beginTransaction();
    
    try {
        // Insert user
        $stmt = $conn->prepare("
            INSERT INTO users (username, email, password_hash, phone, referral_code, referred_by) 
            VALUES (?, ?, ?, ?, ?, ?)
        ");
        $stmt->execute([$username, $email, $password_hash, $phone, $new_referral_code, $referred_by]);
        $user_id = $conn->lastInsertId();
        
        // Create wallet
        $stmt = $conn->prepare("INSERT INTO wallets (user_id, main_balance, bonus_balance) VALUES (?, 0.00, 0.00)");
        $stmt->execute([$user_id]);
        
        // Create referral record if applicable
        if ($referred_by) {
            $stmt = $conn->prepare("INSERT INTO referrals (referrer_id, referred_id, bonus_amount) VALUES (?, ?, 0.00)");
            $stmt->execute([$referred_by, $user_id]);
        }
        
        $conn->commit();
        
        // Generate JWT token
        $token = JWT::encode(['user_id' => $user_id, 'username' => $username]);
        
        success([
            'user' => [
                'id' => $user_id,
                'username' => $username,
                'email' => $email,
                'phone' => $phone,
                'referral_code' => $new_referral_code,
                'referred_by' => $referred_by
            ],
            'token' => $token
        ], 'Registration successful');
        
    } catch (Exception $e) {
        $conn->rollBack();
        throw $e;
    }
    
} catch (Exception $e) {
    error_log("Registration error: " . $e->getMessage());
    error('Registration failed', 'REGISTRATION_ERROR', 500);
}
?>
