<?php
/**
 * User Login API
 * POST /api/auth/login
 */

require_once '../../config/config.php';
require_once '../../config/database.php';
require_once '../../config/jwt.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    error('Method not allowed', 'METHOD_NOT_ALLOWED', 405);
}

try {
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        error('Invalid JSON input', 'INVALID_INPUT');
    }
    
    // Validate required fields
    validateRequired($input, ['username', 'password']);
    
    // Sanitize input
    $username = sanitizeInput($input['username']);
    $password = $input['password'];
    
    $db = new Database();
    $conn = $db->getConnection();
    
    // Find user by username or email
    $stmt = $conn->prepare("
        SELECT id, username, email, password_hash, phone, referral_code, is_active 
        FROM users 
        WHERE (username = ? OR email = ?) AND is_active = 1
    ");
    $stmt->execute([$username, $username]);
    $user = $stmt->fetch();
    
    if (!$user) {
        error('Invalid credentials', 'INVALID_CREDENTIALS', 401);
    }
    
    // Verify password
    if (!verifyPassword($password, $user['password_hash'])) {
        error('Invalid credentials', 'INVALID_CREDENTIALS', 401);
    }
    
    // Get wallet balance
    $stmt = $conn->prepare("SELECT main_balance, bonus_balance FROM wallets WHERE user_id = ?");
    $stmt->execute([$user['id']]);
    $wallet = $stmt->fetch();
    
    // Generate JWT token
    $token = JWT::encode([
        'user_id' => $user['id'],
        'username' => $user['username']
    ]);
    
    success([
        'user' => [
            'id' => $user['id'],
            'username' => $user['username'],
            'email' => $user['email'],
            'phone' => $user['phone'],
            'referral_code' => $user['referral_code'],
            'wallet' => [
                'main_balance' => $wallet['main_balance'] ?? 0,
                'bonus_balance' => $wallet['bonus_balance'] ?? 0
            ]
        ],
        'token' => $token
    ], 'Login successful');
    
} catch (Exception $e) {
    error_log("Login error: " . $e->getMessage());
    error('Login failed', 'LOGIN_ERROR', 500);
}
?>
