<?php
/**
 * Flutterwave Webhook Handler
 * POST /api/webhook/flutterwave
 */

require_once '../../config/config.php';
require_once '../../config/database.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    exit('Method not allowed');
}

try {
    // Get the raw POST data
    $input = file_get_contents('php://input');
    $data = json_decode($input, true);
    
    if (!$data) {
        throw new Exception('Invalid JSON data');
    }
    
    // Verify webhook signature (if provided)
    $signature = $_SERVER['HTTP_VERIF_HASH'] ?? '';
    if (!empty($signature)) {
        $expected_signature = hash_hmac('sha256', $input, FLUTTERWAVE_SECRET_KEY);
        if (!hash_equals($signature, $expected_signature)) {
            throw new Exception('Invalid webhook signature');
        }
    }
    
    // Log webhook for debugging
    error_log("Flutterwave webhook: " . $input);
    
    $event = $data['event'] ?? '';
    $transaction = $data['data'] ?? [];
    
    if ($event === 'charge.completed' && isset($transaction['tx_ref'])) {
        $tx_ref = $transaction['tx_ref'];
        $status = $transaction['status'] ?? '';
        $amount = floatval($transaction['amount'] ?? 0);
        $currency = $transaction['currency'] ?? '';
        
        // Only process TZS transactions
        if ($currency !== 'TZS') {
            error_log("Ignoring non-TZS transaction: " . $currency);
            http_response_code(200);
            exit('OK');
        }
        
        $db = new Database();
        $conn = $db->getConnection();
        
        // Find the transaction
        $stmt = $conn->prepare("
            SELECT t.*, u.id as user_id 
            FROM transactions t
            JOIN users u ON t.user_id = u.id
            WHERE t.reference = ? AND t.type = 'deposit' AND t.status = 'pending'
        ");
        $stmt->execute([$tx_ref]);
        $db_transaction = $stmt->fetch();
        
        if (!$db_transaction) {
            error_log("Transaction not found: " . $tx_ref);
            http_response_code(200);
            exit('OK');
        }
        
        // Start transaction
        $conn->beginTransaction();
        
        try {
            if ($status === 'successful') {
                // Update transaction status
                $stmt = $conn->prepare("
                    UPDATE transactions 
                    SET status = 'completed' 
                    WHERE id = ?
                ");
                $stmt->execute([$db_transaction['id']]);
                
                // Update wallet balance
                $stmt = $conn->prepare("
                    UPDATE wallets 
                    SET main_balance = main_balance + ? 
                    WHERE user_id = ?
                ");
                $stmt->execute([$amount, $db_transaction['user_id']]);
                
                // Check for referral bonus
                $stmt = $conn->prepare("
                    SELECT referred_by FROM users WHERE id = ?
                ");
                $stmt->execute([$db_transaction['user_id']]);
                $user = $stmt->fetch();
                
                if ($user && $user['referred_by']) {
                    // Check if this is the first deposit
                    $stmt = $conn->prepare("
                        SELECT COUNT(*) as deposit_count 
                        FROM transactions 
                        WHERE user_id = ? AND type = 'deposit' AND status = 'completed'
                    ");
                    $stmt->execute([$db_transaction['user_id']]);
                    $deposit_count = $stmt->fetch()['deposit_count'];
                    
                    if ($deposit_count == 1) { // First deposit
                        $bonus_amount = ($amount * REFERRAL_BONUS) / 100;
                        
                        // Add bonus to referrer's wallet
                        $stmt = $conn->prepare("
                            UPDATE wallets 
                            SET bonus_balance = bonus_balance + ? 
                            WHERE user_id = ?
                        ");
                        $stmt->execute([$bonus_amount, $user['referred_by']]);
                        
                        // Create bonus transaction for referrer
                        $stmt = $conn->prepare("
                            INSERT INTO transactions (user_id, type, amount, balance_type, description, status) 
                            VALUES (?, 'referral_bonus', ?, 'bonus', 'Referral bonus from first deposit', 'completed')
                        ");
                        $stmt->execute([$user['referred_by'], $bonus_amount]);
                        
                        // Update referral record
                        $stmt = $conn->prepare("
                            UPDATE referrals 
                            SET bonus_amount = ?, is_paid = 1 
                            WHERE referrer_id = ? AND referred_id = ?
                        ");
                        $stmt->execute([$bonus_amount, $user['referred_by'], $db_transaction['user_id']]);
                    }
                }
                
                $conn->commit();
                error_log("Payment successful: " . $tx_ref . " - Amount: " . $amount);
                
            } else {
                // Payment failed
                $stmt = $conn->prepare("
                    UPDATE transactions 
                    SET status = 'failed' 
                    WHERE id = ?
                ");
                $stmt->execute([$db_transaction['id']]);
                
                $conn->commit();
                error_log("Payment failed: " . $tx_ref);
            }
            
        } catch (Exception $e) {
            $conn->rollBack();
            throw $e;
        }
        
    } else {
        error_log("Unhandled webhook event: " . $event);
    }
    
    http_response_code(200);
    echo 'OK';
    
} catch (Exception $e) {
    error_log("Webhook error: " . $e->getMessage());
    http_response_code(500);
    echo 'Error';
}
?>
