<?php
/**
 * Poll Game State
 * GET /api/game/:id/poll_state
 */

require_once '../../config/config.php';
require_once '../../config/database.php';
require_once '../../config/jwt.php';

if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    error('Method not allowed', 'METHOD_NOT_ALLOWED', 405);
}

try {
    $user_id = getCurrentUser();
    
    // Get game ID from URL
    $path = $_SERVER['REQUEST_URI'];
    preg_match('/\/api\/game\/(\d+)\/poll_state/', $path, $matches);
    $game_id = intval($matches[1] ?? 0);
    
    if (!$game_id) {
        error('Invalid game ID', 'INVALID_GAME_ID');
    }
    
    $db = new Database();
    $conn = $db->getConnection();
    
    // Get game details
    $stmt = $conn->prepare("
        SELECT g.*, u1.username as player1_username, u2.username as player2_username
        FROM games g
        LEFT JOIN users u1 ON g.player1_id = u1.id
        LEFT JOIN users u2 ON g.player2_id = u2.id
        WHERE g.id = ?
    ");
    $stmt->execute([$game_id]);
    $game = $stmt->fetch();
    
    if (!$game) {
        error('Game not found', 'GAME_NOT_FOUND', 404);
    }
    
    // Check if user is part of this game
    if ($game['player1_id'] != $user_id && $game['player2_id'] != $user_id) {
        error('You are not part of this game', 'NOT_PART_OF_GAME', 403);
    }
    
    // Get recent moves
    $stmt = $conn->prepare("
        SELECT m.*, u.username as player_username
        FROM moves m
        JOIN users u ON m.player_id = u.id
        WHERE m.game_id = ?
        ORDER BY m.move_number DESC
        LIMIT 10
    ");
    $stmt->execute([$game_id]);
    $recent_moves = $stmt->fetchAll();
    
    // Determine player number
    $player_number = ($game['player1_id'] == $user_id) ? 1 : 2;
    
    // Load board state
    $board_state = json_decode($game['board_state'] ?? '[]', true);
    
    success([
        'game' => [
            'id' => $game['id'],
            'status' => $game['status'],
            'current_player' => $game['current_player'],
            'player_number' => $player_number,
            'is_your_turn' => $game['current_player'] == $player_number,
            'board' => $board_state,
            'stake_amount' => $game['stake_amount'],
            'winner_id' => $game['winner_id'],
            'is_bot_game' => $game['is_bot_game']
        ],
        'players' => [
            'player1' => [
                'id' => $game['player1_id'],
                'username' => $game['player1_username']
            ],
            'player2' => [
                'id' => $game['player2_id'],
                'username' => $game['player2_username']
            ]
        ],
        'recent_moves' => $recent_moves
    ]);
    
} catch (Exception $e) {
    error_log("Poll state error: " . $e->getMessage());
    error('Failed to get game state', 'POLL_ERROR', 500);
}
?>
